<?php

/**
 * Plugin Name: Interlink Products Manager
 */

if (!defined('ABSPATH')) exit;

define('IPM_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('IPM_PLUGIN_URL', plugin_dir_url(__FILE__));
define('IPM_VERSION', '1.0.0');

require_once IPM_PLUGIN_DIR . 'includes/class-ipm-loader.php';
IPM_Loader::init();

require_once IPM_PLUGIN_DIR . 'includes/support/class-ipm-support.php';
IPM_Support::init();

add_action('init', 'ipm_register_product_category_taxonomy');
function ipm_register_product_category_taxonomy()
{

    register_taxonomy(
        'product_category',
        ['ipm_product'], // ✅ MUST MATCH PRODUCT CPT
        [
            'labels' => [
                'name'          => 'Product Categories',
                'singular_name' => 'Product Category',
            ],
            'public'            => true,   // ✅ frontend access
            'hierarchical'      => true,
            'show_ui'           => false,  // custom admin UI
            'show_admin_column' => false,
            'rewrite'           => ['slug' => 'product-category'],
        ]
    );
}

add_action('wp_ajax_ipm_delete_category', 'ipm_delete_category');
function ipm_delete_category()
{
    check_ajax_referer('ipm_category_nonce');

    $term_id = intval($_POST['term_id'] ?? 0);
    $slug    = sanitize_text_field($_POST['slug'] ?? '');

    if (!$term_id || !$slug) {
        wp_send_json_error('Invalid data');
    }

    wp_delete_term($term_id, 'product_category');
    delete_term_meta($term_id, 'category_image');

    $meta = get_option('interlink_category_list', []);
    unset($meta[$slug]);
    update_option('interlink_category_list', $meta);

    wp_send_json_success();
}

add_action('init', 'ipm_register_product_cpt');
function ipm_register_product_cpt()
{
    register_post_type('ipm_product', [
        'labels' => [
            'name'          => 'Products',
            'singular_name' => 'Product',
        ],

        // 🔥 REQUIRED FOR FRONTEND
        'public'             => true,
        'publicly_queryable' => true,
        'exclude_from_search' => false,

        // Admin control
        'show_ui'            => true,
        'show_in_menu'       => false,

        // URLs & permalinks
        'rewrite' => ['slug' => 'products'],
        'has_archive'        => false,

        'supports' => [
            'title',
            'editor',
            'thumbnail',
        ],
    ]);
}

// Load more products listing
add_action('wp_ajax_ipm_load_more_products', 'ipm_load_more_products');
add_action('wp_ajax_nopriv_ipm_load_more_products', 'ipm_load_more_products');

function ipm_load_more_products()
{

    $paged    = intval($_POST['page'] ?? 1);
    $category = sanitize_text_field($_POST['category'] ?? '');

    $args = [
        'post_type'      => 'ipm_product',
        'posts_per_page' => 6, // 🔥 products per scroll
        'paged'          => $paged,
        'post_status'    => 'publish',
    ];

    if (!empty($category)) {
        $args['tax_query'] = [
            [
                'taxonomy' => 'product_category',
                'field'    => 'slug',
                'terms'    => $category,
            ]
        ];
    }

    $query = new WP_Query($args);

    if (!$query->have_posts()) {
        wp_send_json_error(['end' => true]);
    }

    ob_start();

    while ($query->have_posts()) {
        $query->the_post();
?>
        <div class="ipm-product-card">

            <a href="<?php the_permalink(); ?>" class="ipm-product-image">
                <?php
                if (has_post_thumbnail()) {
                    the_post_thumbnail('medium');
                } else {
                ?>
                    <div class="ipm-product-placeholder">NO IMAGE</div>
                <?php
                }
                ?>
            </a>

            <div class="ipm-product-content">
                <h3 class="ipm-product-title">
                    <a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
                </h3>

                <p class="ipm-product-desc">
                    <?php echo wp_trim_words(get_the_content(), 18); ?>
                </p>

                <a href="<?php the_permalink(); ?>" class="ipm-product-btn">
                    View Details →
                </a>
            </div>

        </div>
<?php
    }

    wp_reset_postdata();

    wp_send_json_success([
        'html' => ob_get_clean(),
        'end'  => false
    ]);
}


// Save categories
add_action('admin_post_ipm_save_categories', 'ipm_handle_category_save');

function ipm_handle_category_save() {
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }
    check_admin_referer('save_category_list', 'category_list_nonce');
    $submitted = $_POST['categories'] ?? [];
    $status_store = [];
    $success_count = 0;
    $seen_slugs = [];
    foreach ($submitted as $cat) {
        if (empty($cat['name'])) continue;
        $name   = sanitize_text_field($cat['name']);
        $desc   = sanitize_textarea_field($cat['description'] ?? '');
        $status = sanitize_text_field($cat['status'] ?? 'active');
        $image  = esc_url_raw($cat['image'] ?? '');
        $slug   = sanitize_title($name);
        if (isset($seen_slugs[$slug])) continue;
        $seen_slugs[$slug] = true;
        $existing = term_exists($slug, 'product_category');
        if (!$existing) {
            $term = wp_insert_term($name, 'product_category', [
                'slug'        => $slug,
                'description' => $desc,
            ]);
            if (is_wp_error($term)) continue;
            $term_id = $term['term_id'];
        } else {
            $term_id = is_array($existing) ? $existing['term_id'] : $existing;
            wp_update_term($term_id, 'product_category', [
                'description' => $desc,
            ]);
        }
        if ($image) {
            update_term_meta($term_id, 'category_image', $image);
        }
        $status_store[$slug] = ['status' => $status];
        $success_count++;
    }
    update_option('interlink_category_list', $status_store);
    wp_redirect(
        add_query_arg(
            ['page' => 'ipm-categories', 'saved' => $success_count],
            admin_url('admin.php')
        )
    );
    exit;
}
// ADD DRAG ORDER SAVE HANDLER (REQUIRED)
add_action('wp_ajax_ipm_save_category_order', 'ipm_save_category_order');

function ipm_save_category_order() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }
    if (!isset($_POST['order']) || !is_array($_POST['order'])) {
        wp_send_json_error('Invalid order');
    }
    $order = [];
    foreach ($_POST['order'] as $slug => $position) {
        $order[sanitize_key($slug)] = intval($position);
    }
    update_option('interlink_category_order', $order);
    wp_send_json_success();
}

// LOCALIZE NONCE FOR JS (REQUIRED)
add_action('admin_enqueue_scripts', function ($hook) {

    if ($hook !== 'products-management_page_ipm-categories') {
        return;
    }

    wp_localize_script(
        'jquery', // using jquery as carrier
        'ipmCategory',
        [
            'nonce' => wp_create_nonce('ipm_category_nonce'),
        ]
    );
});
