<?php
namespace EkobyteCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Ekobyte Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Ekobyte_Blog_Post extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'blogpost';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Blog Post', 'ekobytecore' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'ekobyte-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'ekobytecore' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'ekobytecore' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        /**
         * Layout section
         */
        $this->start_controls_section(
            'ekobyte_post_',
            [
                'label' => esc_html__('Design Layout', 'ekobytecore'),
            ]
        );
        $this->add_control(
            'ekobyte_design_style',
            [
                'label' => esc_html__('Select Layout', 'ekobytecore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'ekobytecore'),
                    'layout-2' => esc_html__('Layout 2', 'ekobytecore'),
                ],
                'default' => 'layout-1',
            ]
        );
        $this->add_control(
            'ekobyte_post__height',
            [
                'label' => esc_html__( 'Height', 'ekobytecore' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .ekobyte-project-img img' => 'height: {{SIZE}}{{UNIT}};object-fit: cover;',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'thumbnail', // // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `thumbnail_size` and `thumbnail_custom_dimension`.
                'exclude' => ['custom'],
                // 'default' => 'ekobyte-post-thumb',
            ]
        );
        $this->add_control(
            'ekobyte_post__pagination',
            [
                'label' => esc_html__( 'Pagination', 'ekobytecore' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'ekobytecore' ),
                'label_off' => esc_html__( 'Hide', 'ekobytecore' ),
                'return_value' => 'yes',
                'default' => 'no',
                'condition' => array(
                    'ekobyte_design_style' => 'layout-1!',
                ),
            ]
        );

        $this->end_controls_section();
               

        /**
         * Title and content
         */
        $this->start_controls_section(
            'ekobyte_section_title',
            [
                'label' => esc_html__('Title & Content', 'ekobytecore'),
            ]
        );
        $this->add_control(
            'ekobyte_section_title_show',
            [
                'label' => esc_html__( 'Section Title & Content', 'ekobytecore' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'ekobytecore' ),
                'label_off' => esc_html__( 'Hide', 'ekobytecore' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        $this->add_control(
            'ekobyte_sub_title',
            [
                'label' => esc_html__('Sub Title', 'ekobytecore'),
                'description' => ekobyte_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Ekobyte Sub Title', 'ekobytecore'),
                'placeholder' => esc_html__('Type Sub Heading Text', 'ekobytecore'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'ekobyte_sub_title_color',
            [
                'label' => __( 'Sub Title Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-title h6' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'ekobyte_title',
            [
                'label' => esc_html__('Title', 'ekobytecore'),
                'description' => ekobyte_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Ekobyte Title Here', 'ekobytecore'),
                'placeholder' => esc_html__('Type Heading Text', 'ekobytecore'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'ekobyte_title_color',
            [
                'label' => __( 'Title Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-title h2' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'ekobyte_title_tag',
            [
                'label' => esc_html__('Title HTML Tag', 'ekobytecore'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'h1' => [
                        'title' => esc_html__('H1', 'ekobytecore'),
                        'icon' => 'eicon-editor-h1'
                    ],
                    'h2' => [
                        'title' => esc_html__('H2', 'ekobytecore'),
                        'icon' => 'eicon-editor-h2'
                    ],
                    'h3' => [
                        'title' => esc_html__('H3', 'ekobytecore'),
                        'icon' => 'eicon-editor-h3'
                    ],
                    'h4' => [
                        'title' => esc_html__('H4', 'ekobytecore'),
                        'icon' => 'eicon-editor-h4'
                    ],
                    'h5' => [
                        'title' => esc_html__('H5', 'ekobytecore'),
                        'icon' => 'eicon-editor-h5'
                    ],
                    'h6' => [
                        'title' => esc_html__('H6', 'ekobytecore'),
                        'icon' => 'eicon-editor-h6'
                    ]
                ],
                'default' => 'h2',
                'toggle' => false,
            ]
        );

        $this->add_responsive_control(
            'ekobyte_align',
            [
                'label' => esc_html__('Alignment', 'ekobytecore'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__('Left', 'ekobytecore'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'ekobytecore'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'ekobytecore'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'left',
                'toggle' => false,
                'selectors' => [
                    '{{WRAPPER}} .ekobyte-sec-box' => 'text-align: {{VALUE}};'
                ]
            ]
        );
        $this->end_controls_section();

        /**
         * Button
         */
        $this->start_controls_section(
            'ekobyte_btn_button_group',
            [
                'label' => esc_html__('Button', 'ekobytecore'),
            ]
        );

        $this->add_control(
            'ekobyte_button_show',
            [
                'label' => esc_html__( 'Show Button', 'ekobytecore' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'ekobytecore' ),
                'label_off' => esc_html__( 'Hide', 'ekobytecore' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'ekobyte_btn_text',
            [
                'label' => esc_html__('Button Text', 'ekobytecore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Button Text', 'ekobytecore'),
                'title' => esc_html__('Enter button text', 'ekobytecore'),
                'label_block' => true,
                'condition' => [
                    'ekobyte_button_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'ekobyte_all_blog_btn_text',
            [
                'label' => esc_html__('All Blog Button Text', 'ekobytecore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('All Blog Button Text', 'ekobytecore'),
                'title' => esc_html__('Enter all blog button text', 'ekobytecore'),
                'label_block' => true,
                'condition' => array(
                    'ekobyte_button_show' => 'yes',
                    'ekobyte_design_style' => 'layout-1',
                ),
            ]
        );
        $this->add_control(
            'ekobyte_all_blog_btn_link',
            [
                'label' => esc_html__('All Blog Button link', 'ekobytecore'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'ekobytecore'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => array(
                    'ekobyte_button_show' => 'yes',
                    'ekobyte_design_style' => 'layout-1',
                ),
                'label_block' => true,
            ]
        );
        $this->end_controls_section();

        
        /**
         * Blog query section
         */
		$this->start_controls_section(
            'ekobyte_post_query',
            [
                'label' => esc_html__('Blog Query', 'ekobytecore'),
            ]
        );

        $post_type = 'post';
        $taxonomy = 'category';

        $this->add_control(
            'posts_per_page',
            [
                'label' => esc_html__('Posts Per Page', 'ekobytecore'),
                'description' => esc_html__('Leave blank or enter -1 for all.', 'ekobytecore'),
                'type' => Controls_Manager::NUMBER,
                'default' => '3',
            ]
        );

        $this->add_control(
            'category',
            [
                'label' => esc_html__('Include Categories', 'ekobytecore'),
                'description' => esc_html__('Select a category to include or leave blank for all.', 'ekobytecore'),
                'type' => Controls_Manager::SELECT2,
                'multiple' => true,
                'options' => ekobyte_get_categories($taxonomy),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'exclude_category',
            [
                'label' => esc_html__('Exclude Categories', 'ekobytecore'),
                'description' => esc_html__('Select a category to exclude', 'ekobytecore'),
                'type' => Controls_Manager::SELECT2,
                'multiple' => true,
                'options' => ekobyte_get_categories($taxonomy),
                'label_block' => true
            ]
        );

        $this->add_control(
            'post__not_in',
            [
                'label' => esc_html__('Exclude Item', 'ekobytecore'),
                'type' => Controls_Manager::SELECT2,
                'options' => ekobyte_get_all_types_post($post_type),
                'multiple' => true,
                'label_block' => true
            ]
        );

        $this->add_control(
            'offset',
            [
                'label' => esc_html__('Offset', 'ekobytecore'),
                'type' => Controls_Manager::NUMBER,
                'default' => '0',
            ]
        );

        $this->add_control(
            'orderby',
            [
                'label' => esc_html__('Order By', 'ekobytecore'),
                'type' => Controls_Manager::SELECT,
                'options' => array(
			        'ID' => 'Post ID',
			        'author' => 'Post Author',
			        'title' => 'Title',
			        'date' => 'Date',
			        'modified' => 'Last Modified Date',
			        'parent' => 'Parent Id',
			        'rand' => 'Random',
			        'comment_count' => 'Comment Count',
			        'menu_order' => 'Menu Order',
			    ),
                'default' => 'date',
            ]
        );

        $this->add_control(
            'order',
            [
                'label' => esc_html__('Order', 'ekobytecore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'asc' 	=> esc_html__( 'Ascending', 'ekobytecore' ),
                    'desc' 	=> esc_html__( 'Descending', 'ekobytecore' )
                ],
                'default' => 'desc',

            ]
        );
        $this->add_control(
            'ignore_sticky_posts',
            [
                'label' => esc_html__( 'Ignore Sticky Posts', 'ekobytecore' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'ekobytecore' ),
                'label_off' => esc_html__( 'No', 'ekobytecore' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'ekobyte_blog_title_word',
            [
                'label' => esc_html__('Title Word Count', 'ekobytecore'),
                'description' => esc_html__('Set how many word you want to display!', 'ekobytecore'),
                'type' => Controls_Manager::NUMBER,
                'default' => '6',
            ]
        );

        $this->add_control(
            'ekobyte_post_content',
            [
                'label' => __('Content', 'ekobytecore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'ekobytecore'),
                'label_off' => __('Hide', 'ekobytecore'),
                'return_value' => 'yes',
                'default' => '',
            ]
        );

        $this->add_control(
            'ekobyte_post_content_limit',
            [
                'label' => __('Content Limit', 'ekobytecore'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => '14',
                'dynamic' => [
                    'active' => true,
                ],
                'condition' => [
                    'ekobyte_post_content' => 'yes'
                ]
            ]
        );

        $this->end_controls_section();


        /**
         * Style section
         */
		$this->start_controls_section(
			'section_style',
			[
				'label' => __( 'Style', 'ekobytecore' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'text_transform',
			[
				'label' => __( 'Text Transform', 'ekobytecore' ),
				'type' => Controls_Manager::SELECT,
				'default' => '',
				'options' => [
					'' => __( 'None', 'ekobytecore' ),
					'uppercase' => __( 'UPPERCASE', 'ekobytecore' ),
					'lowercase' => __( 'lowercase', 'ekobytecore' ),
					'capitalize' => __( 'Capitalize', 'ekobytecore' ),
				],
				'selectors' => [
					'{{WRAPPER}} .title' => 'text-transform: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render the widget ouekobyteut on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		if (get_query_var('paged')) {
            $paged = get_query_var('paged');
        } else if (get_query_var('page')) {
            $paged = get_query_var('page');
        } else {
            $paged = 1;
        }

        // include_categories
        $category_list = '';
        if (!empty($settings['category'])) {
            $category_list = implode(", ", $settings['category']);
        }
        $category_list_value = explode(" ", $category_list);

        // exclude_categories
        $exclude_categories = '';
        if(!empty($settings['exclude_category'])){
            $exclude_categories = implode(", ", $settings['exclude_category']);
        }
        $exclude_category_list_value = explode(" ", $exclude_categories);

        $post__not_in = '';
        if (!empty($settings['post__not_in'])) {
            $post__not_in = $settings['post__not_in'];
            $args['post__not_in'] = $post__not_in;
        }
        $posts_per_page = (!empty($settings['posts_per_page'])) ? $settings['posts_per_page'] : '-1';
        $orderby = (!empty($settings['orderby'])) ? $settings['orderby'] : 'post_date';
        $order = (!empty($settings['order'])) ? $settings['order'] : 'desc';
        $offset_value = (!empty($settings['offset'])) ? $settings['offset'] : '0';
        $ignore_sticky_posts = (! empty( $settings['ignore_sticky_posts'] ) && 'yes' == $settings['ignore_sticky_posts']) ? true : false ;


        // number
        $off = (!empty($offset_value)) ? $offset_value : 0;
        $offset = $off + (($paged - 1) * $posts_per_page);
        $p_ids = array();

        // build up the array
        if (!empty($settings['post__not_in'])) {
            foreach ($settings['post__not_in'] as $p_idsn) {
                $p_ids[] = $p_idsn;
            }
        }

        $args = array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => $posts_per_page,
            'orderby' => $orderby,
            'order' => $order,
            'offset' => $offset,
            'paged' => $paged,
            'post__not_in' => $p_ids,
            'ignore_sticky_posts' => $ignore_sticky_posts
        );

        // exclude_categories
        if ( !empty($settings['exclude_category'])) {

            // Exclude the correct cats from tax_query
            $args['tax_query'] = array(
                array(
                    'taxonomy'	=> 'category',
                    'field'	 	=> 'slug',
                    'terms'		=> $exclude_category_list_value,
                    'operator'	=> 'NOT IN'
                )
            );

            // Include the correct cats in tax_query
            if ( !empty($settings['category'])) {
                $args['tax_query']['relation'] = 'AND';
                $args['tax_query'][] = array(
                    'taxonomy'	=> 'category',
                    'field'		=> 'slug',
                    'terms'		=> $category_list_value,
                    'operator'	=> 'IN'
                );
            }

        } else {
            // Include the cats from $cat_slugs in tax_query
            if (!empty($settings['category'])) {
                $args['tax_query'][] = [
                    'taxonomy' => 'category',
                    'field' => 'slug',
                    'terms' => $category_list_value,
                ];
            }
        }

        $filter_list = $settings['category'];

        // The Query
        $query = new \WP_Query($args); ?>

        <?php if ( $settings['ekobyte_design_style']  == 'layout-2' ):
            $this->add_render_attribute('title_args', 'class', 'title');
        ?>
 
             <!-- Latest Posts Area Start -->
            <div class="latest-posts-area style-1">
                <div class="container">
                    <!-- Section Title Start -->
                    <div class="row">
                        <div class="col-12">
                            <div class="te-section-title justify-content-center text-center">
                                <div class="te-section-content">
                                    <div>
                                        <?php if ( !empty($settings['ekobyte_sub_title']) ) : ?>    
                                            <span class="short-title"><?php echo ekobyte_kses( $settings['ekobyte_sub_title'] ); ?></span>
                                        <?php endif; ?>
                                    </div>
                                    <?php
                                        if ( !empty($settings['ekobyte_title' ]) ) :
                                            printf( '<%1$s %2$s>%3$s</%1$s>',
                                                tag_escape( $settings['ekobyte_title_tag'] ),
                                                $this->get_render_attribute_string( 'title_args' ),
                                                ekobyte_kses( $settings['ekobyte_title' ] )
                                                );
                                        endif;
                                    ?>
                                </div>
                            </div>
                        </div>
                    </div>
                    <!-- Section Title End -->
                    <div class="row">
                        <?php if ($query->have_posts()) : ?>
                            <?php while ($query->have_posts()) : 
                                $query->the_post();
                                global $post;
                                $categories = get_the_category($post->ID);
                            ?>
                                <div class="col-lg-4 col-md-6">
                                    <div class="te-post-card style-2">
                                        <div class="image">
                                            <?php if (has_post_thumbnail( $post->ID ) ): ?>
                                                <img src="<?php the_post_thumbnail_url( $post->ID, $settings['thumbnail_size'] );?>" alt="Post image"/>
                                            <?php endif; ?>
                                            <?php if (has_post_thumbnail( $post->ID ) ): ?>
                                                <div class="te-post-date">
                                                    <span><?php echo get_the_date('d M Y')?></span>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        <div class="content">
                                            <div class="te-post-meta-info">
                                                <div class="te-single-meta">
                                                    <span class="icon">
                                                        <i class="fa-solid fa-user"></i>
                                                    </span>
                                                    <span class="text">By <?php echo get_the_author(); ?></span>
                                                </div>
                                                <div class="te-single-meta">
                                                    <span class="icon">
                                                        <i class="fa-solid fa-comments"></i>
                                                    </span>
                                                    <span class="text">Comments (<?php echo get_comments_number(); ?>)</span>
                                                </div>
                                            </div>
                                            <h3 class="title">
                                                <a href="<?php the_permalink(); ?>"><?php echo wp_trim_words(get_the_title(), $settings['ekobyte_blog_title_word'], ''); ?></a>
                                            </h3>
                                            <div class="te-post-content">
                                                <?php if (!empty($settings['ekobyte_post_content'])):
                                                    $ekobyte_post_content_limit = (!empty($settings['ekobyte_post_content_limit'])) ? $settings['ekobyte_post_content_limit'] : ''; ?>
                                                    <p><?php print wp_trim_words(get_the_excerpt(get_the_ID()), $ekobyte_post_content_limit, ''); ?></p>
                                                <?php endif; ?>
                                            </div>
                                            <?php if (!empty($settings['ekobyte_btn_text'])):?>
                                                <a class="read-btn" href="<?php the_permalink(); ?>"><?php echo ekobyte_kses($settings['ekobyte_btn_text']); ?> <i class="fa-solid fa-arrow-right"></i></a>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endwhile; wp_reset_query(); ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <!-- Latest Posts Area End -->
    
        <?php else:
            $this->add_render_attribute('title_args', 'class', 'title');
        ?>

            <!-- Latest Posts Area Start -->
            <div class="latest-posts-area style-1 background-gradient">
                <div class="container">
                    <div class="row">
                        <div class="col-12">
                            <div class="te-section-title justify-content-center text-center">
                                <div class="te-section-content">
                                    <div>
                                        <?php if ( !empty($settings['ekobyte_sub_title']) ) : ?>    
                                            <span class="short-title"><?php echo ekobyte_kses( $settings['ekobyte_sub_title'] ); ?></span>
                                        <?php endif; ?>
                                    </div>
                                    <?php if ( !empty($settings['ekobyte_title' ]) ) :
                                            printf( '<%1$s %2$s>%3$s</%1$s>',
                                                tag_escape( $settings['ekobyte_title_tag'] ),
                                                $this->get_render_attribute_string( 'title_args' ),
                                                ekobyte_kses( $settings['ekobyte_title' ] )
                                                );
                                        endif;
                                    ?>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <?php if ($query->have_posts()) : ?>
                            <?php while ($query->have_posts()) : 
                                $query->the_post();
                                global $post;
                                $categories = get_the_category($post->ID);
                            ?>
                                <div class="col-lg-4 col-md-6">
                                    <div class="te-post-card style-1">
                                        <div class="image">
                                            <?php if (has_post_thumbnail( $post->ID ) ): ?>
                                                <img src="<?php the_post_thumbnail_url( $post->ID, $settings['thumbnail_size'] );?>" alt="Post image"/>
                                            <?php endif; ?>
                                            <?php if (has_post_thumbnail( $post->ID ) ): ?>
                                                <div class="te-post-date">
                                                    <span><?php echo get_the_date('d M Y')?></span>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        <div class="content">
                                            <div class="te-post-meta-info">
                                                <div class="te-single-meta">
                                                    <span class="icon">
                                                        <i class="fa-solid fa-user"></i>
                                                    </span>
                                                    <span class="text">By <?php echo get_the_author(); ?></span>
                                                </div>
                                                <div class="te-single-meta">
                                                    <span class="icon">
                                                        <i class="fa-solid fa-comments"></i>
                                                    </span>
                                                    <span class="text">Comments (<?php echo get_comments_number(); ?>)</span>
                                                </div>
                                            </div>
                                            <h3 class="title">
                                                <a href="<?php the_permalink(); ?>"><?php echo wp_trim_words(get_the_title(), $settings['ekobyte_blog_title_word'], ''); ?></a>
                                            </h3>
                                            <?php if (!empty($settings['ekobyte_post_content'])):
                                                $ekobyte_post_content_limit = (!empty($settings['ekobyte_post_content_limit'])) ? $settings['ekobyte_post_content_limit'] : '';?>
                                                <p><?php print wp_trim_words(get_the_excerpt(get_the_ID()), $ekobyte_post_content_limit, ''); ?></p>
                                            <?php endif; ?>
                                            <?php if (!empty($settings['ekobyte_btn_text'])):?>
                                                <a class="read-btn" href="<?php the_permalink(); ?>"><?php echo ekobyte_kses($settings['ekobyte_btn_text']); ?> <i class="fa-solid fa-arrow-right"></i></a>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endwhile; wp_reset_query(); ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <!-- Latest Posts Area End -->

    	<?php endif; ?>

       <?php
	}

}

$widgets_manager->register( new Ekobyte_Blog_Post() );