<?php
namespace EkobyteCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Ekobyte Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Ekobyte_About_Talk_To_Us extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'about-talk-to-us';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'About Talk To Us', 'ekobytecore' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'ekobyte-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'ekobytecore' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'ekobytecore' ];
	}


    public function get_ekobyte_contact_form(){
        if ( ! class_exists( 'WPCF7' ) ) {
            return;
        }
        $ekobyte_cfa         = array();
        $ekobyte_cf_args     = array( 'posts_per_page' => -1, 'post_type'=> 'wpcf7_contact_form' );
        $ekobyte_forms       = get_posts( $ekobyte_cf_args );
        $ekobyte_cfa         = ['0' => esc_html__( 'Select Form', 'ekobytecore' ) ];
        if( $ekobyte_forms ){
            foreach ( $ekobyte_forms as $ekobyte_form ){
                $ekobyte_cfa[$ekobyte_form->ID] = $ekobyte_form->post_title;
            }
        }else{
            $ekobyte_cfa[ esc_html__( 'No contact form found', 'ekobytecore' ) ] = 0;
        }
        return $ekobyte_cfa;
    }


	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

		// _ekobyte_image
		$this->start_controls_section(
			'_ekobyte_image',
			[
				'label' => esc_html__('Thumbnail & Phone', 'ekobytecore'),
			]
		);
		$this->add_control(
			'ekobyte_image',
			[
				'label' => esc_html__( 'Choose Image', 'ekobytecore' ),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name' => 'ekobyte_image_size',
				'default' => 'full',
				'exclude' => [
					'custom'
				]
			]
		);
		$this->add_control(
			'ekobyte_image_overlap',
			[
				'label' => esc_html__('Image overlap to top?', 'ekobytecore'),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Yes', 'ekobytecore'),
				'label_off' => esc_html__('No', 'ekobytecore'),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);
		$this->add_responsive_control(
			'ekobyte_image_height',
			[
				'label' => esc_html__( 'Image Height', 'ekobytecore' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .ekobyte-overlap img' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'ekobyte_image_overlap_x',
			[
				'label' => esc_html__( 'Image overlap position', 'ekobytecore' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .ekobyte-overlap img' => 'margin-top: {{SIZE}}{{UNIT}};',
				],
				'condition' => array(
					'ekobyte_image_overlap' => 'yes',
				),
			]
		);
		
		// single icon or image
		$this->add_control(
            'ekobyte_single_icon_type',
            [
                'label' => esc_html__('Select Icon Type', 'ekobytecore'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'image',
                'options' => [
                    'image' => esc_html__('Image', 'ekobytecore'),
                    'icon' => esc_html__('Icon', 'ekobytecore'),
                ],
            ]
        );

        $this->add_control(
            'ekobyte_icon_image',
            [
                'label' => esc_html__('Upload Icon Image', 'ekobytecore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'ekobyte_single_icon_type' => 'image'
                ]

            ]
        );

        if (ekobyte_is_elementor_version('<', '2.6.0')) {
            $this->add_control(
                'icon',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICON,
                    'label_block' => true,
                    'default' => 'fa fa-star',
                    'condition' => [
                        'ekobyte_single_icon_type' => 'icon'
                    ]
                ]
            );
        } else {
            $this->add_control(
                'selected_icon',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICONS,
                    'fa4compatibility' => 'icon',
                    'label_block' => true,
                    'default' => [
                        'value' => 'far fa-star',
                        'library' => 'regular',
                    ],
                    'condition' => [
                        'ekobyte_single_icon_type' => 'icon'
                    ]
                ]
            );
        }

		$this->add_control(
			'ekobyte_talk_to_us_heading',
			[
				'label' => esc_html__('Talk to us heading', 'ekobytecore'),
				'description' => ekobyte_get_allowed_html_desc( 'intermediate' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__('Need Help?', 'ekobytecore'),
				'placeholder' => esc_html__('Type text before phone number', 'ekobytecore'),
				'label_block' => true,
			]
		);

		$this->add_control(
            'ekobyte_talk_to_us_heading_color',
            [
                'label' => __( 'Heading Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .talk-to-us .left-numbber .text-content h4' => 'color: {{VALUE}}',
                ],
            ]
        );

		$this->add_control(
			'ekobyte_talk_to_us_phone',
			[
				'label' => esc_html__('Phone Number', 'ekobytecore'),
				'description' => ekobyte_get_allowed_html_desc( 'intermediate' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__('111-123123', 'ekobytecore'),
				'placeholder' => esc_html__('Type your phone number', 'ekobytecore'),
				'label_block' => true,
			]
		);

		$this->add_control(
            'ekobyte_talk_to_us_phone_color',
            [
                'label' => __( 'Phone Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .talk-to-us .left-numbber .text-content h3' => 'color: {{VALUE}}',
                ],
            ]
        );

		$this->end_controls_section();

		// form title and sub title
		$this->start_controls_section(
			'ekobyte_section_title',
			[
				'label' => esc_html__('Form Title & Sub Title', 'ekobytecore'),
			]
		);

		$this->add_control(
			'ekobyte_section_title_show',
			[
				'label' => esc_html__( 'Section Title & Content', 'ekobytecore' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'ekobytecore' ),
				'label_off' => esc_html__( 'Hide', 'ekobytecore' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->add_control(
			'ekobyte_sub_title',
			[
				'label' => esc_html__('Sub Title', 'ekobytecore'),
				'description' => ekobyte_get_allowed_html_desc( 'intermediate' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__('Ekobyte Sub Title', 'ekobytecore'),
				'placeholder' => esc_html__('Type Sub Title Text', 'ekobytecore'),
				'label_block' => true,
			]
		);

		$this->add_control(
            'ekobyte_sub_title_color',
            [
                'label' => __( 'Sub Title Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-title h6' => 'color: {{VALUE}}',
                ],
            ]
        );

		$this->add_control(
			'ekobyte_title',
			[
				'label' => esc_html__('Title', 'ekobytecore'),
				'description' => ekobyte_get_allowed_html_desc( 'intermediate' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__('Ekobyte Title Here', 'ekobytecore'),
				'placeholder' => esc_html__('Type Heading Text', 'ekobytecore'),
				'label_block' => true,
			]
		);

		$this->add_control(
            'ekobyte_title_color',
            [
                'label' => __( 'Title Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-title h2' => 'color: {{VALUE}}',
                ],
            ]
        );

		$this->add_control(
			'ekobyte_title_tag',
			[
				'label' => esc_html__('Title HTML Tag', 'ekobytecore'),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'h1' => [
						'title' => esc_html__('H1', 'ekobytecore'),
						'icon' => 'eicon-editor-h1'
					],
					'h2' => [
						'title' => esc_html__('H2', 'ekobytecore'),
						'icon' => 'eicon-editor-h2'
					],
					'h3' => [
						'title' => esc_html__('H3', 'ekobytecore'),
						'icon' => 'eicon-editor-h3'
					],
					'h4' => [
						'title' => esc_html__('H4', 'ekobytecore'),
						'icon' => 'eicon-editor-h4'
					],
					'h5' => [
						'title' => esc_html__('H5', 'ekobytecore'),
						'icon' => 'eicon-editor-h5'
					],
					'h6' => [
						'title' => esc_html__('H6', 'ekobytecore'),
						'icon' => 'eicon-editor-h6'
					]
				],
				'default' => 'h2',
				'toggle' => false,
			]
		);

		$this->add_responsive_control(
			'ekobyte_align',
			[
				'label' => esc_html__('Alignment', 'ekobytecore'),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__('Left', 'ekobytecore'),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__('Center', 'ekobytecore'),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__('Right', 'ekobytecore'),
						'icon' => 'eicon-text-align-right',
					],
				],
				'default' => 'left',
				'toggle' => false,
				'selectors' => [
					'{{WRAPPER}}' => 'text-align: {{VALUE}};'
				]
			]
		);
		$this->end_controls_section();

		// form
        $this->start_controls_section(
            'ekobytecore_contact',
            [
                'label' => esc_html__('Contact Form', 'ekobytecore'),
            ]
        );

        $this->add_control(
            'ekobytecore_select_contact_form',
            [
                'label'   => esc_html__( 'Select Form', 'ekobytecore' ),
                'type'    => Controls_Manager::SELECT,
                'default' => '0',
                'options' => $this->get_ekobyte_contact_form(),
            ]
        );

        $this->end_controls_section();

		$this->start_controls_section(
			'section_style',
			[
				'label' => __( 'Style', 'ekobytecore' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'text_transform',
			[
				'label' => __( 'Text Transform', 'ekobytecore' ),
				'type' => Controls_Manager::SELECT,
				'default' => '',
				'options' => [
					'' => __( 'None', 'ekobytecore' ),
					'uppercase' => __( 'UPPERCASE', 'ekobytecore' ),
					'lowercase' => __( 'lowercase', 'ekobytecore' ),
					'capitalize' => __( 'Capitalize', 'ekobytecore' ),
				],
				'selectors' => [
					'{{WRAPPER}} .title' => 'text-transform: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render the widget ouekobyteut on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		if ( !empty($settings['ekobyte_image']['url']) ) {
			$ekobyte_image = !empty($settings['ekobyte_image']['id']) ? wp_get_attachment_image_url( $settings['ekobyte_image']['id'], $settings['ekobyte_image_size_size']) : $settings['ekobyte_image']['url'];
			$ekobyte_image_alt = get_post_meta($settings["ekobyte_image"]["id"], "_wp_attachment_image_alt", true);
		}	
		?>

			<section class="talk-to-us v2">
                <div class="container">
                    <div class="row">
                        <div class="col-lg-6">
                            <div class="left-img-numbber">
                                <div class="left-img">
								<?php if ($settings['ekobyte_image']['url'] || $settings['ekobyte_image']['id']) : ?>  
                                    <img src="<?php echo esc_url($ekobyte_image); ?>" alt="<?php echo esc_attr($ekobyte_image_alt); ?>">
                                <?php endif; ?>
                                </div>
                                <div class="left-numbber">
                                    <div class="my-icon">
                                        <?php if($settings['ekobyte_single_icon_type'] !== 'image') : ?>
                                            <?php if (!empty($settings['icon']) || !empty($settings['selected_icon']['value'])) : ?>
                                                <?php ekobyte_render_icon($settings, 'icon', 'selected_icon'); ?>
                                            <?php endif; ?>   
                                        <?php else : ?>                                
                                            <?php if (!empty($settings['ekobyte_icon_image']['url'])): ?>  
                                                <img src="<?php echo $settings['ekobyte_icon_image']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($settings['ekobyte_icon_image']['url']), '_wp_attachment_image_alt', true); ?>">
                                            <?php endif; ?> 
                                        <?php endif; ?> 
                                    </div>
                                    <div class="text-content">
                                        <?php if ( !empty($settings['ekobyte_talk_to_us_heading']) ) : ?>    
        									<h4><?php echo ekobyte_kses( $settings['ekobyte_talk_to_us_heading'] ); ?></h4>
        								<?php endif; ?>
                                        <?php if ( !empty($settings['ekobyte_left_section_phone_number']) ) : ?>
                                            <h3><a href="tel:<?php echo esc_attr(str_replace(' ', '-', $settings['ekobyte_left_section_phone_number'])); ?>"><?php echo ekobyte_kses( $settings['ekobyte_left_section_phone_number'] ); ?></a></h3>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-6">
                            <div class="right-content">
                                <div class="section-title">
								<?php if ( !empty($settings['ekobyte_sub_title']) ) : ?>    
									<h6><?php echo ekobyte_kses( $settings['ekobyte_sub_title'] ); ?></h6>
								<?php endif; ?>

								<?php
									if ( !empty($settings['ekobyte_title' ]) ) :
										printf( '<%1$s %2$s>%3$s</%1$s>',
											tag_escape( $settings['ekobyte_title_tag'] ),
											$this->get_render_attribute_string( 'title' ),
											ekobyte_kses( $settings['ekobyte_title' ] )
											);
									endif;
								?>
								</div>

								<?php if( !empty($settings['ekobytecore_select_contact_form']) ) : ?> 
								<div class="message-form"> 
									<?php echo do_shortcode( '[contact-form-7  id="'.$settings['ekobytecore_select_contact_form'].'"]' ); ?> 
								</div> 
								<?php else : ?>
									<?php echo '<div class="alert alert-info"><p class="m-0">' . __('Please Select contact form.', 'ekobytecore' ). '</p></div>'; ?>
								<?php endif; ?>
								
                            </div>
                        </div>
                    </div>
                </div>
            </section>

        <?php 
	}
}

$widgets_manager->register( new Ekobyte_About_Talk_To_Us() );