<?php
namespace EkobyteCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Repeater;
use \Elementor\Utils;
use \Elementor\Control_Media;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Ekobyte Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Ekobyte_Company_Benefit extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'company-benefit';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Company Benefit', 'ekobytecore' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'ekobyte-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'ekobytecore' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'ekobytecore' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {


        // ekobyte_section_title
        $this->start_controls_section(
            'ekobyte_section_title',
            [
                'label' => esc_html__('Title & Content', 'ekobytecore'),
            ]
        );

        $this->add_control(
            'ekobyte_sub_title',
            [
                'label' => esc_html__('Sub Title', 'ekobytecore'),
                'description' => ekobyte_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Ekobyte Sub Title', 'ekobytecore'),
                'placeholder' => esc_html__('Type Sub Heading Text', 'ekobytecore'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'ekobyte_sub_title_color',
            [
                'label' => __( 'Sub Title Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-title h6' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'ekobyte_title',
            [
                'label' => esc_html__('Title', 'ekobytecore'),
                'description' => ekobyte_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Ekobyte Title Here', 'ekobytecore'),
                'placeholder' => esc_html__('Type Heading Text', 'ekobytecore'),
                'label_block' => true,
            ]
        ); 
        
        $this->add_control(
            'ekobyte_title_color',
            [
                'label' => __( 'Title Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-title h2' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'ekobyte_description',
            [
                'label' => esc_html__('Description', 'ekobytecore'),
                'description' => ekobyte_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('Ekobyte section description here', 'ekobytecore'),
                'placeholder' => esc_html__('Type section description here', 'ekobytecore'),
            ]
        );

        $this->add_control(
            'ekobyte_description_color',
            [
                'label' => __( 'Description Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-title p' => 'color: {{VALUE}}',
                ],
            ]
        );
 

        $this->add_control(
            'ekobyte_title_tag',
            [
                'label' => esc_html__('Title HTML Tag', 'ekobytecore'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'h1' => [
                        'title' => esc_html__('H1', 'ekobytecore'),
                        'icon' => 'eicon-editor-h1'
                    ],
                    'h2' => [
                        'title' => esc_html__('H2', 'ekobytecore'),
                        'icon' => 'eicon-editor-h2'
                    ],
                    'h3' => [
                        'title' => esc_html__('H3', 'ekobytecore'),
                        'icon' => 'eicon-editor-h3'
                    ],
                    'h4' => [
                        'title' => esc_html__('H4', 'ekobytecore'),
                        'icon' => 'eicon-editor-h4'
                    ],
                    'h5' => [
                        'title' => esc_html__('H5', 'ekobytecore'),
                        'icon' => 'eicon-editor-h5'
                    ],
                    'h6' => [
                        'title' => esc_html__('H6', 'ekobytecore'),
                        'icon' => 'eicon-editor-h6'
                    ]
                ],
                'default' => 'h2',
                'toggle' => false,
            ]
        );

        $this->add_responsive_control(
            'ekobyte_align',
            [
                'label' => esc_html__('Alignment', 'ekobytecore'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'text-left' => [
                        'title' => esc_html__('Left', 'ekobytecore'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'text-center' => [
                        'title' => esc_html__('Center', 'ekobytecore'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'text-right' => [
                        'title' => esc_html__('Right', 'ekobytecore'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'left',
                'toggle' => false,
            ]
        );
        $this->end_controls_section();

        // Features group
        $this->start_controls_section(
            'ekobyte_features',
            [
                'label' => esc_html__('Features List', 'ekobytecore'),
                'description' => esc_html__( 'Control all the style settings from Style tab', 'ekobytecore' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new \Elementor\Repeater();

                $repeater->add_control(
            'ekobyte_service_icon_type',
            [
                'label' => esc_html__('Select Icon Type', 'ekobytecore'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'image',
                'options' => [
                    'image' => esc_html__('Image', 'ekobytecore'),
                    'icon' => esc_html__('Icon', 'ekobytecore'),
                ],
            ]
        );

        $repeater->add_control(
            'ekobyte_icon_image',
            [
                'label' => esc_html__('Upload Icon Image', 'ekobytecore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'ekobyte_service_icon_type' => 'image'
                ]

            ]
        );

        if (ekobyte_is_elementor_version('<', '2.6.0')) {
            $repeater->add_control(
                'icon',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICON,
                    'label_block' => true,
                    'default' => 'fa fa-star',
                    'condition' => [
                        'ekobyte_service_icon_type' => 'icon'
                    ]
                ]
            );
        } else {
            $repeater->add_control(
                'selected_icon',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICONS,
                    'fa4compatibility' => 'icon',
                    'label_block' => true,
                    'default' => [
                        'value' => 'far fa-star',
                        'library' => 'regular',
                    ],
                    'condition' => [
                        'ekobyte_service_icon_type' => 'icon'
                    ]
                ]
            );
        }
        
        $repeater->add_control(
            'ekobyte_features_title', [
                'label' => esc_html__('Title', 'ekobytecore'),
                'description' => ekobyte_get_allowed_html_desc( 'basic' ),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('Feature Title', 'ekobytecore'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'ekobyte_features_title_color',
            [
                'label' => __( 'Title Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .company-benefits .content-box .benefits-list .list-content h4' => 'color: {{VALUE}}',
                ],
            ]
        );

        $repeater->add_control(
            'ekobyte_features_text', [
                'label' => esc_html__('Features Description', 'ekobytecore'),
                'description' => ekobyte_get_allowed_html_desc( 'basic' ),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => esc_html__('Feature Description', 'ekobytecore'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'ekobyte_features_text_color',
            [
                'label' => __( 'Text Color', 'ekobytecore' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .company-benefits .content-box .benefits-list .list-content p' => 'color: {{VALUE}}',
                ],
            ]
        );
     
        $this->add_control(
            'ekobyte_features_list',
            [
                'label' => esc_html__('Features - List', 'ekobytecore'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'ekobyte_features_title' => esc_html__('Discover', 'ekobytecore'),
                    ],
                    [
                        'ekobyte_features_title' => esc_html__('Define', 'ekobytecore')
                    ]
                ],
                'title_field' => '{{{ ekobyte_features_title }}}',
            ]
        );
        $this->end_controls_section();


        // ekobyte_btn_button_group
        $this->start_controls_section(
            'ekobyte_btn_button_group',
            [
                'label' => esc_html__('Button', 'ekobytecore'),
            ]
        );

        $this->add_control(
            'ekobyte_btn_button_show',
            [
                'label' => esc_html__( 'Show Button', 'ekobytecore' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'ekobytecore' ),
                'label_off' => esc_html__( 'Hide', 'ekobytecore' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'ekobyte_btn_text',
            [
                'label' => esc_html__('Button Text', 'ekobytecore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Button Text', 'ekobytecore'),
                'title' => esc_html__('Enter button text', 'ekobytecore'),
                'label_block' => true,
                'condition' => [
                    'ekobyte_btn_button_show' => 'yes'
                ],
            ]
        );
        $this->add_control(
            'ekobyte_btn_link_type',
            [
                'label' => esc_html__('Button Link Type', 'ekobytecore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
                'condition' => [
                    'ekobyte_btn_button_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'ekobyte_btn_link',
            [
                'label' => esc_html__('Button link', 'ekobytecore'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'ekobytecore'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'ekobyte_btn_link_type' => '1',
                    'ekobyte_btn_button_show' => 'yes'
                ],
                'label_block' => true,
            ]
        );
        $this->add_control(
            'ekobyte_btn_page_link',
            [
                'label' => esc_html__('Select Button Page', 'ekobytecore'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => ekobyte_get_all_pages(),
                'condition' => [
                    'ekobyte_btn_link_type' => '2',
                    'ekobyte_btn_button_show' => 'yes'
                ]
            ]
        );
        $this->end_controls_section();

        // _ekobyte_image
		$this->start_controls_section(
            '_ekobyte_image',
            [
                'label' => esc_html__('Thumbnail', 'ekobytecore'),
            ]
        );
       
        $this->add_control(
            'ekobyte_right_side_image',
            [
                'label' => esc_html__( 'Choose Right Side Image', 'ekobytecore' ),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );
    
        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'ekobyte_image_size',
                'default' => 'full',
                'exclude' => [
                    'custom'
                ]
            ]
        );
        $this->add_control(
            'ekobyte_image_overlap',
            [
                'label' => esc_html__('Image overlap to top?', 'ekobytecore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ekobytecore'),
                'label_off' => esc_html__('No', 'ekobytecore'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );
        $this->add_responsive_control(
            'ekobyte_image_height',
            [
                'label' => esc_html__( 'Image Height', 'ekobytecore' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .ekobyte-overlap img' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'ekobyte_image_overlap_x',
            [
                'label' => esc_html__( 'Image overlap position', 'ekobytecore' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .ekobyte-overlap img' => 'margin-top: {{SIZE}}{{UNIT}};',
                ],
                'condition' => array(
                    'ekobyte_image_overlap' => 'yes',
                ),
            ]
        );
        $this->end_controls_section();


		$this->start_controls_section(
			'section_style',
			[
				'label' => __( 'Style', 'ekobytecore' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'text_transform',
			[
				'label' => __( 'Text Transform', 'ekobytecore' ),
				'type' => Controls_Manager::SELECT,
				'default' => '',
				'options' => [
					'' => __( 'None', 'ekobytecore' ),
					'uppercase' => __( 'UPPERCASE', 'ekobytecore' ),
					'lowercase' => __( 'lowercase', 'ekobytecore' ),
					'capitalize' => __( 'Capitalize', 'ekobytecore' ),
				],
				'selectors' => [
					'{{WRAPPER}} .title' => 'text-transform: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render the widget ouekobyteut on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

            if ( !empty($settings['ekobyte_right_side_image']['url']) ) {
                $ekobyte_right_side_image = !empty($settings['ekobyte_right_side_image']['id']) ? wp_get_attachment_image_url( $settings['ekobyte_right_side_image']['id'], $settings['ekobyte_image_size_size']) : $settings['ekobyte_right_side_image']['url'];
                $ekobyte_right_side_image_alt = get_post_meta($settings["ekobyte_right_side_image"]["id"], "_wp_attachment_image_alt", true);
            }            

			$this->add_render_attribute('title_args', 'class', 'section-title');

            // Link
            if ('2' == $settings['ekobyte_btn_link_type']) {
                $this->add_render_attribute('ekobyte-button-arg', 'href', get_permalink($settings['ekobyte_btn_page_link']));
                $this->add_render_attribute('ekobyte-button-arg', 'target', '_self');
                $this->add_render_attribute('ekobyte-button-arg', 'rel', 'nofollow');
                $this->add_render_attribute('ekobyte-button-arg', 'class', 'link-anime v2');
            } else {
                if ( ! empty( $settings['ekobyte_btn_link']['url'] ) ) {
                    $this->add_link_attributes( 'ekobyte-button-arg', $settings['ekobyte_btn_link'] );
                    $this->add_render_attribute('ekobyte-button-arg', 'class', 'link-anime v2');
                }
            }
        ?>

        <section class="company-benefits">
            <div class="container">
                <div class="row">
                    <div class="col-lg-6">
                        <div class="content-box">
                            <div class="section-title">
                                <?php if ( !empty($settings['ekobyte_sub_title']) ) : ?>    
                                    <h6><?php echo ekobyte_kses( $settings['ekobyte_sub_title'] ); ?></h6>
                                <?php endif; ?>
                                <?php
                                    if ( !empty($settings['ekobyte_title' ]) ) :
                                        printf( '<%1$s %2$s>%3$s</%1$s>',
                                            tag_escape( $settings['ekobyte_title_tag'] ),
                                            $this->get_render_attribute_string( 'title_args' ),
                                            ekobyte_kses( $settings['ekobyte_title' ] )
                                            );
                                    endif;
                                ?>
                                <?php if ( !empty($settings['ekobyte_description']) ) : ?>
                                    <p><?php echo ekobyte_kses( $settings['ekobyte_description'] ); ?></p>
                                <?php endif; ?>
                            </div>
                            <div class="benefits-list">
                                <ul>
                                    <?php foreach ($settings['ekobyte_features_list'] as $item) : ?>
                                        <li>
                                            <div class="my-icon">
                                                <?php if($item['ekobyte_service_icon_type'] !== 'image') : ?>
                                                    <?php if (!empty($item['icon']) || !empty($item['selected_icon']['value'])) : ?>
                                                        <?php ekobyte_render_icon($item, 'icon', 'selected_icon'); ?>
                                                    <?php endif; ?>   
                                                <?php else : ?>                                
                                                    <?php if (!empty($item['ekobyte_icon_image']['url'])): ?>  
                                                        <img src="<?php echo $item['ekobyte_icon_image']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['ekobyte_icon_image']['url']), '_wp_attachment_image_alt', true); ?>">
                                                    <?php endif; ?> 
                                                <?php endif; ?> 
                                            </div>
                                            <div class="list-content">
                                                <h4><?php echo ekobyte_kses($item['ekobyte_features_title' ]); ?></h4>
                                                <p><?php echo ekobyte_kses($item['ekobyte_features_text' ]); ?></p>
                                            </div>
                                        </li>
                                    <?php endforeach; ?>
                                </ul>
                                
                                <?php if (!empty($settings['ekobyte_btn_text'])) : ?>
                                    <a <?php echo $this->get_render_attribute_string( 'ekobyte-button-arg' ); ?>>
                                        <?php echo $settings['ekobyte_btn_text']; ?>
                                    </a>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-6">
                        <?php if ($settings['ekobyte_right_side_image']['url'] || $settings['ekobyte_right_side_image']['id']) : ?>
                            <div class="box-img">
                                <img src="<?php echo esc_url($ekobyte_right_side_image); ?>" alt="<?php echo esc_attr($ekobyte_right_side_image_alt); ?>">
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </section>

        <?php 
	}
}

$widgets_manager->register( new Ekobyte_Company_Benefit() );